/*
 * statistics module - script interface to internal statistics manager
 *
 * Copyright (C) 2006 Voice Sistem S.R.L.
 *
 * This file is part of Kamailio, a free SIP server.
 *
 * SPDX-License-Identifier: GPL-2.0-or-later
 *
 * Kamailio is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version
 *
 * Kamailio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

/*!
 * \defgroup statistics The statistics module
 */
/*!
 * \brief Script interface
 * \ingroup statistics
 * \author bogdan
 */


#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "../../core/sr_module.h"
#include "../../core/dprint.h"
#include "../../core/ut.h"
#include "../../core/mod_fix.h"
#include "../../core/kemi.h"
#include "../../core/counters.h"
#include "../../core/mem/mem.h"
#include "stats_funcs.h"

MODULE_VERSION

static int reg_param_stat(modparam_t type, void *val);
static int mod_init(void);
static int w_update_stat(struct sip_msg *msg, char *stat, char *n);
static int w_reset_stat(struct sip_msg *msg, char *stat, char *foo);
static int fixup_stat(void **param, int param_no);

struct stat_or_pv
{
	stat_var *stat;
	pv_spec_t *pv;
};

struct long_or_pv
{
	long val;
	pv_spec_t *pv;
};

/* clang-format off */
static cmd_export_t cmds[] = {
	{"update_stat", (cmd_function)w_update_stat, 2, fixup_stat, 0, ANY_ROUTE},
	{"reset_stat", (cmd_function)w_reset_stat, 1, fixup_stat, 0,
			REQUEST_ROUTE | BRANCH_ROUTE | FAILURE_ROUTE | ONREPLY_ROUTE | LOCAL_ROUTE},
	{0, 0, 0, 0, 0, 0}
};

static param_export_t mod_params[] = {
	{"variable", PARAM_STRING | PARAM_USE_FUNC, (void *)reg_param_stat},
	{0, 0, 0}
};

struct module_exports exports = {
	"statistics",    /* module name */
	DEFAULT_DLFLAGS, /* dlopen flags */
	cmds,            /* exported functions */
	mod_params,      /* exported parameters */
	0,               /* exported rpc functions */
	0,               /* exported pseudo-variables */
	0,               /* response handling function */
	mod_init,        /* module init function */
	0,               /* per-child init function */
	0                /* module destroy function */
};
/* clang-format on */

static int reg_param_stat(modparam_t type, void *val)
{
	return reg_statistic((char *)val);
}


static int mod_init(void)
{
	if(register_all_mod_stats() != 0) {
		LM_ERR("failed to register statistic variables\n");
		return E_UNSPEC;
	}
	return 0;
}


static int fixup_stat(void **param, int param_no)
{
	struct stat_or_pv *sopv;
	struct long_or_pv *lopv;
	str s;
	long n = 0;
	int err = 0;

	s.s = (char *)*param;
	s.len = strlen(s.s);
	if(param_no == 1) {
		/* var name - string or pv */
		sopv = (struct stat_or_pv *)pkg_malloc(sizeof(struct stat_or_pv));
		if(sopv == NULL) {
			PKG_MEM_ERROR;
			return E_OUT_OF_MEM;
		}
		memset(sopv, 0, sizeof(struct stat_or_pv));
		/* is it pv? */
		if(s.s[0] == '$') {
			if(fixup_pvar_null(param, 1) != 0) {
				LM_ERR("invalid pv %s as parameter\n", s.s);
				pkg_free(sopv);
				return E_CFG;
			}
			sopv->pv = (pv_spec_t *)(*param);
		} else {
			/* it is string */
			sopv->stat = get_stat(&s);
			if(sopv->stat == 0) {
				LM_ERR("variable <%s> not defined\n", s.s);
				pkg_free(sopv);
				return E_CFG;
			}
		}
		*param = (void *)sopv;
		return 0;
	} else if(param_no == 2) {
		lopv = (struct long_or_pv *)pkg_malloc(sizeof(struct long_or_pv));
		if(lopv == NULL) {
			PKG_MEM_ERROR;
			return E_OUT_OF_MEM;
		}
		memset(lopv, 0, sizeof(struct long_or_pv));
		/* is it pv? */
		if(s.s[0] == '$') {
			if(fixup_pvar_pvar(param, 2) != 0) {
				LM_ERR("invalid pv %s as parameter\n", s.s);
				pkg_free(lopv);
				return E_CFG;
			}
			lopv->pv = (pv_spec_t *)(*param);
		} else {
			/* it is string */
			/* update value - integer */
			if(s.s[0] == '-' || s.s[0] == '+') {
				n = str2s(s.s + 1, s.len - 1, &err);
				if(s.s[0] == '-')
					n = -n;
			} else {
				n = str2s(s.s, s.len, &err);
			}
			lopv->val = n;
		}

		if(err == 0) {
			if(n == 0
					&& (s.s[0]
							!= '$')) { //we can't check the value of the pvar so have to ignore this check if it is a pvar
				LM_ERR("update with 0 has no sense\n");
				pkg_free(lopv);
				return E_CFG;
			}
			*param = (void *)lopv;
			return 0;
		} else {
			LM_ERR("bad update number <%s>\n", (char *)(*param));
			pkg_free(lopv);
			return E_CFG;
		}
	}
	return 0;
}


static int w_update_stat(struct sip_msg *msg, char *stat_p, char *long_p)
{
	struct stat_or_pv *sopv = (struct stat_or_pv *)stat_p;
	struct long_or_pv *lopv = (struct long_or_pv *)long_p;
	pv_value_t pv_val;
	stat_var *stat;
	long n = 0;
	int err;

	if(lopv->val) {
		n = lopv->val;
	} else {
		if(pv_get_spec_value(msg, lopv->pv, &pv_val) != 0
				|| (pv_val.flags & PV_VAL_STR) == 0) {
			LM_ERR("failed to get pv string value\n");
			return -1;
		}
		str s = pv_val.rs;
		/* it is string */
		/* update value - integer */
		if(s.s[0] == '-' || s.s[0] == '+') {
			n = str2s(s.s + 1, s.len - 1, &err);
			if(s.s[0] == '-')
				n = -n;
		} else {
			n = str2s(s.s, s.len, &err);
		}
	}

	if(sopv->stat) {
		update_stat(sopv->stat, (long)n);
	} else {
		if(pv_get_spec_value(msg, sopv->pv, &pv_val) != 0
				|| (pv_val.flags & PV_VAL_STR) == 0) {
			LM_ERR("failed to get pv string value\n");
			return -1;
		}
		stat = get_stat(&(pv_val.rs));
		if(stat == 0) {
			LM_ERR("variable <%.*s> not defined\n", pv_val.rs.len, pv_val.rs.s);
			return -1;
		}
		update_stat(stat, (long)n);
	}

	return 1;
}

static int ki_update_stat(sip_msg_t *msg, str *sname, int sval)
{
	stat_var *stat;

	stat = get_stat(sname);
	if(stat == 0) {
		LM_ERR("variable <%.*s> not defined\n", sname->len, sname->s);
		return -1;
	}
	update_stat(stat, (long)sval);
	return 1;
}

static int w_reset_stat(struct sip_msg *msg, char *stat_p, char *foo)
{
	struct stat_or_pv *sopv = (struct stat_or_pv *)stat_p;
	pv_value_t pv_val;
	stat_var *stat;

	if(sopv->stat) {
		reset_stat(sopv->stat);
	} else {
		if(pv_get_spec_value(msg, sopv->pv, &pv_val) != 0
				|| (pv_val.flags & PV_VAL_STR) == 0) {
			LM_ERR("failed to get pv string value\n");
			return -1;
		}
		stat = get_stat(&(pv_val.rs));
		if(stat == 0) {
			LM_ERR("variable <%.*s> not defined\n", pv_val.rs.len, pv_val.rs.s);
			return -1;
		}
		reset_stat(stat);
	}

	return 1;
}

static int ki_reset_stat(sip_msg_t *msg, str *sname)
{
	stat_var *stat;

	stat = get_stat(sname);
	if(stat == 0) {
		LM_ERR("variable <%.*s> not defined\n", sname->len, sname->s);
		return -1;
	}
	reset_stat(stat);
	return 1;
}

/**
 *
 */
/* clang-format off */
static sr_kemi_t sr_kemi_statistics_exports[] = {
	{ str_init("statistics"), str_init("update_stat"),
		SR_KEMIP_INT, ki_update_stat,
		{ SR_KEMIP_STR, SR_KEMIP_INT, SR_KEMIP_NONE,
			SR_KEMIP_NONE, SR_KEMIP_NONE, SR_KEMIP_NONE }
	},
	{ str_init("statistics"), str_init("reset_stat"),
		SR_KEMIP_INT, ki_reset_stat,
		{ SR_KEMIP_STR, SR_KEMIP_NONE, SR_KEMIP_NONE,
			SR_KEMIP_NONE, SR_KEMIP_NONE, SR_KEMIP_NONE }
	},

	{ {0, 0}, {0, 0}, 0, NULL, { 0, 0, 0, 0, 0, 0 } }
};
/* clang-format on */

/**
 *
 */
int mod_register(char *path, int *dlflags, void *p1, void *p2)
{
	sr_kemi_modules_add(sr_kemi_statistics_exports);
	return 0;
}
