/**
 * \file mzcbor2mzmlcli.cpp
 * \date 04/12/2025
 * \author Olivier Langella
 * \brief CLI tool to convert mzCBOR to mzML
 */


/*******************************************************************************
 * Copyright (c) 2025 Olivier Langella <Olivier.Langella@universite-paris-saclay.fr>.
 *
 * This file is part of the PAPPSOms++ library.
 *
 *     PAPPSOms++ is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     PAPPSOms++ is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with PAPPSOms++.  If not, see <http://www.gnu.org/licenses/>.
 *
 ******************************************************************************/


#include "mzcbor2mzmlcli.h"
#include <QIODevice>
#include <QCommandLineParser>
#include "pappsomspp/core/processing/uimonitor/uimonitortext.h"
#include "pappsomspp/core/processing/uimonitor/uimonitorvoid.h"
#include "pappsomspp/core/processing/cbor/mzcbor/converttomzml.h"
#include "pappsomspp/core/pappsoexception.h"
#include <QDateTime>
#include <QFile>
#include <QFileInfo>
#include <QTimer>
#include <QDir>
#include <sys/stat.h>


Mzcbor2MzmlCli::Mzcbor2MzmlCli(QObject *parent) : QObject(parent)
{
  // get the instance of the main application
  app = QCoreApplication::instance();
  // setup everything here
  // create any global objects
  // setup debug and warning mode
}

Mzcbor2MzmlCli::~Mzcbor2MzmlCli()
{
}

// 10ms after the application starts this method will run
// all QT messaging is running at this point so threads, signals and slots
// will all work as expected.
void
Mzcbor2MzmlCli::run()
{
  qSetMessagePattern(QString("%{file}@%{line}, %{function}(): %{message}"));

  QTextStream errorStream(stderr, QIODevice::WriteOnly);
  QTextStream outputStream(stderr, QIODevice::WriteOnly);

  try
    {
      qDebug();
      QCommandLineParser parser;

      // throw pappso::PappsoException("test");
      // throw pappso::PappsoException("test");
      parser.setApplicationDescription(QString("mzcbor2mzml")
                                         .append(" ")
                                         .append(PAPPSOMSPP_VERSION)
                                         .append(" mzCBOR to mzML converter"));

      parser.addHelpOption();
      parser.addVersionOption();

      QCommandLineOption mzcborInputOption(
        QStringList() << "i" << "input",
        QCoreApplication::translate(
          "mzCBOR", "path to the mzCBOR input file <mzcbor> or use '-' to read data from stdin."),
        QCoreApplication::translate("mzcbor", "mzcbor"));

      QCommandLineOption mzmlOutputOption(
        QStringList() << "o" << "output",
        QCoreApplication::translate("mzML file", "PSI mzML output file <mzML>"),
        QCoreApplication::translate("mzML", "mzML"));


      parser.addOption(mzcborInputOption);
      parser.addOption(mzmlOutputOption);

      qDebug();

      // Process the actual command line arguments given by the user
      parser.process(*app);

      // QCoreApplication * app(this);
      // Add your main code here
      // qDebug();
      const QDateTime dt_begin = QDateTime::currentDateTime();
      const QStringList args   = parser.positionalArguments();


      qDebug();
      pappso::UiMonitorText monitor(outputStream);


      // PsmProcess cbor_psm_process(json_parameters);

      bool is_dir = false;
      QFile input_mzcbor_file;
      if(!parser.isSet(mzcborInputOption))
        {
          if(parser.positionalArguments().contains("-"))
            {
              input_mzcbor_file.open(stdin, QIODevice::ReadOnly);
            }
          else
            {
              throw pappso::PappsoException(QObject::tr(
                "input file is not set. Please provide --input or use - to read from <stdin>"));
            }
        }
      else
        {

          QFileInfo input_mzml_fileinfo(parser.value(mzcborInputOption));
          if(input_mzml_fileinfo.isFile())
            {
              input_mzcbor_file.setFileName(parser.value(mzcborInputOption));
              if(!input_mzcbor_file.open(QIODevice::ReadOnly))
                {
                  throw pappso::PappsoException(
                    QObject::tr("Unable to open mzCBOR file '%1' :\n%2\n")
                      .arg(QFileInfo(input_mzcbor_file).absoluteFilePath())
                      .arg(input_mzcbor_file.errorString()));
                }
            }
          else if(input_mzml_fileinfo.isDir())
            {
              qDebug();
              is_dir = true;
              convertAllMzcborInDir(monitor, input_mzml_fileinfo, parser.value(mzmlOutputOption));
            }
        }

      if(is_dir)
        {
        }
      else
        {
          QFile mzml_file;
          QXmlStreamWriter *p_mzmlWriter = nullptr;
          QString mzcbor_index_filename;
          if(parser.isSet(mzmlOutputOption))
            {
              QString mzml_filename = parser.value(mzmlOutputOption);
              if(!mzml_filename.isEmpty())
                {
                  mzml_file.setFileName(mzml_filename);
                  mzml_file.open(QIODevice::WriteOnly);
                  p_mzmlWriter = new QXmlStreamWriter(&mzml_file);
                }
            }
          else
            {
              mzml_file.open(stdout, QIODevice::WriteOnly);
              p_mzmlWriter = new QXmlStreamWriter(&mzml_file);
            }


          if(p_mzmlWriter != nullptr)
            {
              pappso::cbor::mzcbor::ConvertToMzml mzml_convert(p_mzmlWriter);

              mzml_convert.readCbor(&input_mzcbor_file, monitor);
              input_mzcbor_file.close();
              mzml_file.close();
            }
        }
      qDebug();
    }
  catch(pappso::PappsoException &error)
    {

      errorStream << QString("Oops! an error occurred in %1. Don't Panic :\n%2")
                       .arg(QCoreApplication::applicationName())
                       .arg(error.qwhat());

      errorStream << Qt::endl << Qt::endl;

      errorStream.flush();
      app->exit(1);
      exit(1);
    }

  catch(std::exception &error)
    {

      errorStream << QString("Oops! an error occurred in %1. Don't Panic :\n%2")
                       .arg(QCoreApplication::applicationName())
                       .arg(error.what());

      errorStream << Qt::endl << Qt::endl;

      errorStream.flush();
      app->exit(1);
      exit(1);
    }


  // you must call quit when complete or the program will stay in the
  // messaging loop
  quit();
}

// call this routine to quit the application
void
Mzcbor2MzmlCli::quit()
{
  // you can do some cleanup here
  // then do emit finished to signal CoreApplication to quit
  emit finished();
}

// shortly after quit is called the CoreApplication will signal this routine
// this is a good place to delete any objects that were created in the
// constructor and/or to stop any threads
void
Mzcbor2MzmlCli::aboutToQuitApp()
{
  // stop threads
  // sleep(1);   // wait for threads to stop.
  // delete any objects
}


void
Mzcbor2MzmlCli::convertAllMzcborInDir(pappso::UiMonitorInterface &monitor,
                                      const QFileInfo &input_dir,
                                      const QString &output_str)
{
  qDebug();
  QStringList mzcbor_file_list;
  QDir dir(input_dir.absoluteDir());
  dir.setFilter(QDir::Files);
  foreach(QFileInfo fileInfo, dir.entryInfoList(QDir::Files))
    {

      qDebug() << fileInfo.suffix().toLower();
      if(fileInfo.isFile() && (fileInfo.suffix().toLower() == "mzcbor"))
        {
          mzcbor_file_list << fileInfo.absoluteFilePath();
        }
    }
  QDir output_dir(dir);
  if(!output_str.isEmpty())
    {
      output_dir.setPath(QFileInfo(output_str).absoluteDir().absolutePath());
    }
  if(!output_dir.exists())
    {

      throw pappso::PappsoException(QObject::tr("ERROR the output directory does not exists :\n%1")
                                      .arg(output_dir.absolutePath()));
    }

  for(QString &input_file_name : mzcbor_file_list)
    {
      QString output_file_name =
        output_dir.absoluteFilePath(QFileInfo(input_file_name).baseName().append(".mzML"));
      qDebug() << output_file_name;
      qDebug() << input_file_name;

      monitor.setStatus(
        QObject::tr("Converting %1 to %2").arg(input_file_name).arg(output_file_name));
      QFile input_file(input_file_name);
      if(input_file.open(QIODevice::ReadOnly))
        {

          QFile mzml_file;
          mzml_file.setFileName(output_file_name);
          if(mzml_file.open(QIODevice::WriteOnly))
            {
              QXmlStreamWriter *p_xmlWriter = new QXmlStreamWriter(&mzml_file);

              pappso::UiMonitorVoid void_monitor;

              pappso::cbor::mzcbor::ConvertToMzml mzml_convert(p_xmlWriter);

              mzml_convert.readCbor(&input_file, void_monitor);
              input_file.close();
              mzml_file.close();
              delete p_xmlWriter;
            }
        }
    }
}

int
main(int argc, char **argv)
{
  umask(0);
  // QTextStream consoleErr(stderr);
  // QTextStream consoleOut(stdout, QIODevice::WriteOnly);
  // ConsoleOut::setCout(new QTextStream(stdout, QIODevice::WriteOnly));
  // ConsoleOut::setCerr(new QTextStream(stderr, QIODevice::WriteOnly));
  qDebug();
  QCoreApplication app(argc, argv);
  qDebug();
  QCoreApplication::setApplicationName("mzcbor2mzml");
  QCoreApplication::setApplicationVersion(PAPPSOMSPP_VERSION);
  QLocale::setDefault(QLocale::system());

  // create the main class
  Mzcbor2MzmlCli myMain;
  // connect up the signals
  QObject::connect(&myMain, SIGNAL(finished()), &app, SLOT(quit()));
  QObject::connect(&app, SIGNAL(aboutToQuit()), &myMain, SLOT(aboutToQuitApp()));
  qDebug();


  // This code will start the messaging engine in QT and in
  // 10ms it will start the execution in the MainClass.run routine;
  QTimer::singleShot(10, &myMain, SLOT(run()));
  return app.exec();
}
