use core:io;
use parser;

/**
 * Representation of an HTTP request.
 */
class Request {
	// Method verb.
	Method method;

	// HTTP version used.
	Version version = Version:HTTP_1_0;

	// Path requested, including URL parameters. The path is absolute if the `host` header was
	// given, otherwise it is relative. The host header is retained in the headers array even if it
	// is duplicated in the url.
	QueryUrl path;

	// Headers. All keys are lowercased.
	private Str->Str headers;

	// Cookies (automatically parsed from headers).
	Str->Str cookies;

	// Data in the body.
	Buffer body;

	// Response indicated by the parsing logic.
	Status immediateResponse = Status:none;

	// Create.
	init() {
		init {}
	}

	// Shorthand to create a request in an error state.
	init(Status immediateResponse) {
		init { immediateResponse = immediateResponse; }
	}

	// Create with basic information.
	init(Method method, Version version, Url path) {
		init {
			method = method;
			version = version;
			path = path;
		}
	}

	// Create, parsing the query string into a path.
	init(Method method, RequestComponents path) {
		init {
			method = method;
			version = path.version;
			path = path.path;
		}
	}

	// Set header.
	assign header(Str key, Str val) {
		headers.put(key.toAsciiLowercase, val);
	}

	// Set header.
	assign header(Buffer key, Buffer val) {
		headers.put(key.toAsciiLowercase, val.fromUtf8);
	}

	// Get header.
	Str? header(Str key) {
		return headers.at(key.toAsciiLowercase);
	}

	// Get header, assume it exists.
	Str getHeader(Str key) {
		unless (r = header(key))
			throw HttpError("Header ${key} does not exist.");
		r;
	}

	// Called by the parser to finalize parsing. This lets us update the path based on the presence of the host header.
	void finalize() {
		if (path.absolute) {
			// Nothing to do.
		} else if (host = headers.at("host")) {
			path = httpUrl(host).push(path);
		}
	}


	// To string.
	void toS(StrBuf to) : override {
		to << "request {";
		{
			Indent z(to);
			to << "\nmethod: " << method;
			to << "\nversion: " << version;
			to << "\npath: " << path;
			to << "\nheaders: " << headers;
			to << "\ncookies: " << cookies;
			to << "\nimmediate response: " << immediateResponse;
			to << "\nbody:\n";
			Indent w(to);
			to << body;
		}
		to << "\n}";
	}

	// Write to a stream.
	void write(OStream to) {
		Utf8Output text(to, windowsTextInfo(false));
		text.autoFlush = false;

		// TODO: How to handle url encoding of the URL while keeping parameters handled?
		text.write(if (version == Version:HTTP_0_9) { "GET"; } else { method.toS; });
		text.write(" ");
		for (i, part in path) {
			if (i == 0)
				if (path.absolute)
					continue;
			text.write("/");
			text.write(part.escapeUrl);
		}

		// Write query parameters.
		if (path.parameters.any) {
			text.write("?");
			Bool first = true;
			for (k, v in path.parameters) {
				if (!first)
					text.write("&");
				first = false;
				text.write(escapeUrlParam(k));
				text.write("=");
				text.write(escapeUrlParam(v));
			}
		}

		if (version == Version:HTTP_0_9) {
			// Nothing for HTTP 0.9.
			text.write("\n");
			text.flush();
			return;
		} else if (version == Version:HTTP_1_0) {
			text.write(" HTTP/1.0\n");
		} else if (version == Version:HTTP_1_1) {
			text.write(" HTTP/1.1\n");
		} else {
			throw HttpError("Unsupported HTTP version: ${version}");
		}

		// Write the host header.
		if (path.absolute) {
			text.write("Host: ");
			text.write(path[0]);
			text.write("\n");
		}

		// Write headers:
		for (k, v in headers) {
			if (k == "content-length")
				continue;
			if (path.absolute & k == "host")
				continue;
			text.write(k);
			text.write(": ");
			text.write(v);
			text.write("\n");
		}

		// Write cookies.
		if (cookies.any) {
			// Todo: one or multiple cookie items?
			text.write("Cookie:");
			for (k, v in cookies) {
				text.write(" ");
				text.write(k);
				text.write("=");
				text.write(v);
			}
			text.write("\n");
		}

		// See if we should write a body:
		if (body.any) {
			text.write("Content-Length: ");
			text.write(body.filled.toS);
			text.write("\n\n");
			text.flush();

			to.write(body);
		} else {
			text.write("\n\n");
			text.flush();
		}
	}
}


parseRequestHeader : parser(recursive descent, binary) {
	start = SHTTPReq;

	optional delimiter = SOptDelimiter;
	required delimiter = SReqDelimiter;

	void SOptDelimiter();
	SOptDelimiter : " *";

	void SReqDelimiter();
	SReqDelimiter : " +";

	Request SHTTPReq();
	SHTTPReq => r : SReqLine r - SHeaders(me) - SFinalize(me);

	void SFinalize(Request r);
	SFinalize => finalize(r) : ;

	void SNewline();
	SNewline : "\r\n";

	Request SReqLine();
	SReqLine => Request(method, pathVer) : SMethod method ~ SReqPathVer pathVer;

	Method SMethod();
	SMethod => Method.GET() : "GET";
	SMethod => Method.POST() : "POST";
	SMethod => Method.PUT() : "PUT";
	SMethod => Method.DELETE() : "DELETE";
	SMethod => Method.HEAD() : "HEAD";
	SMethod => Method.OPTIONS() : "OPTIONS";

	RequestComponents SReqPathVer();
	SReqPathVer => RequestComponents() : "/" - SReqSegments(me);
	SReqPathVer => RequestComponents() : "*" - SVersion -> version;

	void SReqSegments(RequestComponents c);
	SReqSegments => c : SVersion -> version;
	SReqSegments => c : "?" - SQuery(c) - SVersion -> version;
	SReqSegments => c : "[^/? \n\r]+" -> push - "/?" - SReqSegments(c);

	void SQuery(RequestComponents c);
	SQuery : SQueryEntry(c) - ("&" - SQuery(c))?;

	void SQueryEntry(RequestComponents c);
	SQueryEntry => push(c, k, v) : "[^=\r\n]+" k, "=", "[^ &\r\n]+" v;

	Version SVersion();
	SVersion => Version.HTTP_0_9() : "\r\n";
	SVersion => Version.HTTP_0_9() : " *HTTP/0.9\r\n"; // This is actually not how it is represented...
	SVersion => Version.HTTP_1_0() : " *HTTP/1.0\r\n";
	SVersion => Version.HTTP_1_1() : " *HTTP/1.1\r\n";

	void SHeaders(Request req);
	SHeaders : SNewline;
	SHeaders[1] : "Cookie", ":" ~ SCookies(req);
	SHeaders[0] => header(req, k, v) : "[^:\r\n]+" k - ":" ~ "[^\r\n]+" v - SNewline - SHeaders(req);

	void SCookies(Request req);
	SCookies : SNewline;
	SCookies => addCookie(req, k, v) : "[^=]+" k - "=" - "[^;\r\n]+" v - " *" - ("; *" - SCookies(req))?;
}

private void addCookie(Request to, Buffer key, Buffer val) {
	to.cookies.put(key.fromUtf8(), val.fromUtf8());
}

/**
 * Class used to store URL components and pass them to Request.
 */
class RequestComponents {
	// Version.
	Version version;

	// Pieces of the query string.
	Str[] pieces;

	// URL parameters.
	Str->Str params;

	// Create an URL.
	QueryUrl path() { QueryUrl(pieces, params); }

	// Push a piece.
	void push(Buffer piece) {
		pieces << unescapeUrl(piece);
	}

	// Push a parameter.
	void push(Buffer key, Buffer value) {
		params.put(unescapeUrlParam(key), unescapeUrlParam(value));
	}

	// Set version.
	void version(Version v) { version = v; }
}
